<?php
// FILE: login.php (PERFORMANCE OPTIMIZED)
ini_set('display_errors', 1);
error_reporting(E_ALL);
date_default_timezone_set('Asia/Tokyo');

require_once __DIR__ . '/config.php';
session_start([
	'cookie_samesite' => 'None',
	'cookie_secure' => true,
	'cookie_httponly' => true
]);
require_once __DIR__ . '/functions.php';
require_once __DIR__ . '/db.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
	header('Location: index.php');
	exit;
}
validate_csrf_token();
$pdo = get_pdo();

try {
	// --- 1. POSTデータを受け取る ---
	$user_name = trim($_POST['user_name'] ?? '');
	$room_id_action = trim($_POST['room_id'] ?? '');
	$new_room_name = trim($_POST['new_room_name'] ?? '');
	$description = trim($_POST['description'] ?? '');
	$usertype = trim($_POST['usertype'] ?? '');
	$height = trim($_POST['height'] ?? '');
	$weight = trim($_POST['weight'] ?? '');
	$age = trim($_POST['age'] ?? '');
	$areaname = trim($_POST['areaname'] ?? '');
	$tastes = trim($_POST['tastes'] ?? '');
	$wants = trim($_POST['wants'] ?? '');
	$hopes = trim($_POST['hopes'] ?? '');
	$beep_enabled = ($_POST['beep_enabled'] ?? '0') === '1' ? 1 : 0;
	$delete_logs_on_logout = ($_POST['delete_logs_on_logout'] ?? '0') === '1' ? 1 : 0;
	$log_expire_days = (int)($_POST['log_expire_days'] ?? 1);
	$is_spectator = ($_POST['is_spectator'] ?? '0') === '1' ? 1 : 0;
	$flag_meet = ($_POST['flag_meet'] ?? '0') === '1' ? 1 : 0;
	$flag_chatplay = ($_POST['flag_chatplay'] ?? '0') === '1' ? 1 : 0;
	$flag_onlyreal = ($_POST['flag_onlyreal'] ?? '0') === '1' ? 1 : 0;
	$flag_friend = ($_POST['flag_friend'] ?? '0') === '1' ? 1 : 0;
	$auto_leave_on_close = 1;

	if ($user_name === '管理人') {
		$log_expire_days = -1;
	}

	if (empty($user_name) || empty($room_id_action)) {
		throw new Exception('ユーザー名とルームを選択してください。');
	}
	if ($room_id_action === 'new_room' && empty($new_room_name)) {
		throw new Exception('新しいルーム名を入力してください。');
	}

	$pdo->beginTransaction();

	// --- 2. ユーザーIDを特定または新規作成 ---
	$user_id = null;
	$unique_user_id = $_COOKIE[COOKIE_NAME] ?? null;
	if ($unique_user_id) {
		$stmt_user = $pdo->prepare("SELECT user_id, deny, denied_at FROM users3 WHERE unique_user_id = :unique_id");
		$stmt_user->execute([':unique_id' => $unique_user_id]);
		$user = $stmt_user->fetch();
		if ($user) {
			$user_id = $user['user_id'];
			if ($user['deny'] == 1) {
				$reset_minutes = defined('RESET_DENY_MINUTE') ? RESET_DENY_MINUTE : 5;
				if ($user['denied_at'] && (strtotime($user['denied_at']) < time() - ($reset_minutes * 60))) {
					$pdo->prepare("UPDATE users3 SET deny = 0, denied_at = NULL WHERE user_id = :user_id")->execute([':user_id' => $user_id]);
				} else {
					$remaining_time = $reset_minutes * 60 - (time() - strtotime($user['denied_at']));
					$remaining_minutes = ceil($remaining_time / 60);
					throw new Exception("このアカウントは一時的に利用が制限されています。あと約{$remaining_minutes}分後に再度お試しください。");
				}
			}
		}
	}
	if (!$user_id) {
		$unique_user_id = uniqid(rand(), true);
		$pdo->prepare("INSERT INTO users3 (unique_user_id, user_name, host_ip, last_login_date) VALUES (:unique_id, :user_name, :host_ip, NOW())")
		  ->execute([':unique_id' => $unique_user_id, ':user_name' => $user_name, ':host_ip' => $_SERVER['REMOTE_ADDR']]);
		$user_id = $pdo->lastInsertId();
		setcookie(COOKIE_NAME, $unique_user_id, ['expires' => time() + 31536000, 'path' => '/', 'secure' => true, 'httponly' => true, 'samesite' => 'None']);
	}

	// --- 3. ルームIDと名前を確定 & 必要なルーム更新 ---
	$final_room_id = null;
	$final_room_name = '';
	if ($room_id_action === 'new_room') {
		$final_room_name = $new_room_name;
		$pdo->prepare("INSERT INTO chat_rooms (room_name, created_by, description, started_at, flag_meet, flag_chatplay, flag_onlyreal, flag_friend) VALUES (:name, :uid, :desc, NOW(), :fm, :fc, :fo, :ff)")
		  ->execute([':name' => $final_room_name, ':uid' => $user_id, ':desc' => $description, ':fm' => $flag_meet, ':fc' => $flag_chatplay, ':fo' => $flag_onlyreal, ':ff' => $flag_friend]);
		$final_room_id = $pdo->lastInsertId();
	} else {
		$final_room_id = (int)$room_id_action;
		$stmt_room = $pdo->prepare("SELECT room_name, is_locked, created_by FROM chat_rooms WHERE room_id = :room_id");
		$stmt_room->execute([':room_id' => $final_room_id]);
		$room = $stmt_room->fetch();
		if (!$room) { throw new Exception('ルームが存在しません。'); }
		if ($room['is_locked'] == 1 && $user_id != $room['created_by']) { throw new Exception('このルームは入室が制限されています。'); }
		$final_room_name = $room['room_name'];
		
		$stmt_count = $pdo->prepare("SELECT COUNT(*) FROM online_status WHERE room_id = :room_id AND last_active > DATE_SUB(NOW(), INTERVAL :threshold SECOND)");
		$stmt_count->execute([':room_id' => $final_room_id, ':threshold' => ONLINE_THRESHOLD_SECONDS]);
		if ($stmt_count->fetchColumn() == 0) {
			$pdo->prepare("UPDATE chat_rooms SET created_by = :uid, description = :desc, started_at = NOW(), flag_meet = :fm, flag_chatplay = :fc, flag_onlyreal = :fo, flag_friend = :ff WHERE room_id = :rid")
			  ->execute([':uid' => $user_id, ':desc' => $description, ':fm' => $flag_meet, ':fc' => $flag_chatplay, ':fo' => $flag_onlyreal, ':ff' => $flag_friend, ':rid' => $final_room_id]);
		}
	}

	// --- 4. ユーザープロファイルと状態を一度のクエリで更新 ---
	$update_fields = [
		':user_name' => $user_name,
		':usertype' => $usertype,
		':height' => $height,
		':weight' => $weight,
		':age' => $age,
		':areaname' => $areaname,
		':tastes' => $tastes,
		':wants' => $wants,
		':hopes' => $hopes,
		':profile_message' => $description,
		':flag_meet' => $flag_meet,
		':flag_chatplay' => $flag_chatplay,
		':flag_onlyreal' => $flag_onlyreal,
		':flag_friend' => $flag_friend,
		':host_ip' => $_SERVER['REMOTE_ADDR'],
		':user_agent' => trim($_SERVER['HTTP_USER_AGENT'] ?? ''),
		':device_type' => get_device_type(trim($_SERVER['HTTP_USER_AGENT'] ?? '')),
		':beep_enabled' => $beep_enabled,
		':log_expire_days' => $log_expire_days,
		':is_spectator' => $is_spectator,
		':last_active_room_id' => $final_room_id,
		':user_id' => $user_id
	];
	$set_parts = [];
	foreach ($update_fields as $key => $value) {
		if ($key !== ':user_id') $set_parts[] = ltrim($key, ':') . " = {$key}";
	}
	$sql_update_profile = "UPDATE users3 SET last_login_date = NOW(), " . implode(', ', $set_parts) . " WHERE user_id = :user_id";
	$pdo->prepare($sql_update_profile)->execute($update_fields);

	// --- 5. オンライン状態を記録 ---
	$pdo->prepare("INSERT INTO online_status (user_id, room_id, last_active, message_cnt) VALUES (:uid, :rid, NOW(), 0) ON DUPLICATE KEY UPDATE last_active = NOW(), message_cnt = 0")
	  ->execute([':uid' => $user_id, ':rid' => $final_room_id]);

	// --- 6. 入室ログを記録 ---
	if ($is_spectator == 0) {
		$pdo->prepare("INSERT INTO chat_log3 (room_id, message, is_system) VALUES (:rid, :msg, 1)")
		  ->execute([':rid' => $final_room_id, ':msg' => h($user_name) . 'さんが入室しました']);
	}

	$pdo->commit();
	
	// --- 7. キャッシュクリアとリダイレクト ---
	clear_room_list_cache();
	touch_user_list_update_timestamp($final_room_id);

	$_SESSION['user_id'] = $user_id;
	$_SESSION['user_name'] = $user_name;
	$_SESSION['room_id'] = $final_room_id;
	$_SESSION['room_name'] = $final_room_name;
	
	header('Location: chat.php');
	exit();

} catch (Exception $e) {
	if (isset($pdo) && $pdo->inTransaction()) { $pdo->rollBack(); }
	
	$_SESSION['error'] = $e->getMessage();
	error_log("Login Error: " . $e->getMessage() . " in " . $e->getFile() . " on line " . $e->getLine());
	header('Location: index.php');
	exit();
}