<?php

// FILE: functions.php
// XSS対策
function h($str)
{
    return htmlspecialchars((string)$str, ENT_QUOTES, 'UTF-8');
}

// === CSRF対策関数 ===
function generate_csrf_token()
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        return;
    }
    if (empty($_SESSION['csrf_token'])) {
        $token = bin2hex(random_bytes(32));
        $_SESSION['csrf_token'] = $token;
    }
}

function csrf_token_field()
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        return '<input type="hidden" name="csrf_token" value="">';
    }
    if (empty($_SESSION['csrf_token'])) {
        generate_csrf_token();
    }
    $token = $_SESSION['csrf_token'] ?? '';
    return '<input type="hidden" name="csrf_token" value="' . h($token) . '">';
}

function validate_csrf_token()
{
    if (session_status() !== PHP_SESSION_ACTIVE) {
        error_log("CSRF validation skipped: Session not active.");
        return;
    }
    $post_token = $_POST['csrf_token'] ?? '';
    $session_token = $_SESSION['csrf_token'] ?? null;
    if (empty($post_token) || empty($session_token) || !hash_equals($session_token, $post_token)) {
        $_SESSION['error'] = 'ページの有効期限が切れました。もう一度お試しください。';
        unset($_SESSION['csrf_token']);
        header('Location: index.php');
        exit;
    }
}

function resize_image($source_path, $destination_path, $options = [])
{
    $max_width = $options['max_width'] ?? null;
    $max_height = $options['max_height'] ?? null;
    if (is_null($max_width) && is_null($max_height)) {
        return copy($source_path, $destination_path);
    }
    $image_info = @getimagesize($source_path);
    if ($image_info === false) {
        error_log("getimagesize failed for: " . $source_path);
        return false;
    }
    list($width, $height, $type) = $image_info;
    $should_resize = false;
    if (!is_null($max_width) && $width > $max_width) {
        $should_resize = true;
    }
    if (!is_null($max_height) && $height > $max_height) {
        $should_resize = true;
    }
    if (!$should_resize) {
        return copy($source_path, $destination_path);
    }
    $ratio = $width / $height;
    $new_width = $width;
    $new_height = $height;
    if (!is_null($max_height) && $new_height > $max_height) {
        $new_height = $max_height;
        $new_width = $new_height * $ratio;
    }
    if (!is_null($max_width) && $new_width > $max_width) {
        $new_width = $max_width;
        $new_height = $new_width / $ratio;
    }
    $new_width = floor($new_width);
    $new_height = floor($new_height);
    $thumb = imagecreatetruecolor($new_width, $new_height);
    $source = null;
    switch ($type) {
        case IMAGETYPE_JPEG: $source = @imagecreatefromjpeg($source_path); break;
        case IMAGETYPE_PNG:
            $source = @imagecreatefrompng($source_path);
            imagealphablending($thumb, false);
            imagesavealpha($thumb, true);
            break;
        case IMAGETYPE_GIF: $source = @imagecreatefromgif($source_path); break;
        default: error_log("Unsupported image type for resizing: " . $type); return false;
    }
    if (!$source) {
        error_log("imagecreatefrom... failed for: " . $source_path);
        return false;
    }
    imagecopyresampled($thumb, $source, 0, 0, 0, 0, $new_width, $new_height, $width, $height);
    $success = false;
    switch ($type) {
        case IMAGETYPE_JPEG: $success = imagejpeg($thumb, $destination_path, 85); break;
        case IMAGETYPE_PNG: $success = imagepng($thumb, $destination_path, 6); break;
        case IMAGETYPE_GIF: $success = imagegif($thumb, $destination_path); break;
    }
    imagedestroy($source);
    imagedestroy($thumb);
    return $success;
}

function clear_room_list_cache()
{
    $cache_file = __DIR__ . '/cache/room_list_cache.json';
    if (file_exists($cache_file)) {
        @unlink($cache_file);
    }
}

function get_device_type($user_agent)
{
    if (preg_match('/(tablet|ipad|playbook)|(android(?!.*(mobi|opera mini)))/i', $user_agent)) {
        return 'Tablet';
    }
    if (preg_match('/(up.browser|up.link|mmp|symbian|smartphone|midp|wap|phone|android|iemobile|mobile|opera mini)/i', $user_agent)) {
        return 'Mobile';
    }
    return 'PC';
}

function get_user_list_update_timestamp($room_id)
{
    $cache_file = __DIR__ . '/cache/user_list_update_' . (int)$room_id . '.txt';
    if (file_exists($cache_file)) {
        return @file_get_contents($cache_file);
    }
    return '0';
}

function touch_user_list_update_timestamp($room_id)
{
    $cache_dir = __DIR__ . '/cache';
    if (!is_dir($cache_dir)) {
        @mkdir($cache_dir, 0755, true);
    }
    $cache_file = $cache_dir . '/user_list_update_' . (int)$room_id . '.txt';
    @file_put_contents($cache_file, time());
}

/**
 * 定期メンテナンス処理を実行する
 * @param PDO $pdo データベース接続オブジェクト
 * @return bool 成功した場合は true、失敗した場合は false
 */
function run_maintenance(PDO $pdo)
{
    try {
        $pdo->beginTransaction();

        // ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼ 以下のブロックをコメントアウトまたは削除 ▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼▼
        // /*
        // 1. 300秒 (5分)以上アクティブでないオンライン状態を削除
        // この処理が競合状態を引き起こすため、get_rooms.phpからは実行しない
        $current_user_id = 0;
        if (isset($_COOKIE[COOKIE_NAME])) {
            $stmt_uid = $pdo->prepare("SELECT user_id FROM users3 WHERE unique_user_id = :unique_id");
            $stmt_uid->execute([':unique_id' => $_COOKIE[COOKIE_NAME]]);
            $user_id_result = $stmt_uid->fetchColumn();
            if ($user_id_result) {
                $current_user_id = (int)$user_id_result;
            }
        }
        $inactive_threshold = date('Y-m-d H:i:s', time() - 300);
        $stmt_delete = $pdo->prepare("DELETE FROM online_status WHERE last_active < :threshold AND user_id != :user_id");
        $stmt_delete->execute([':threshold' => $inactive_threshold, ':user_id' => $current_user_id]);
        // */
        // ▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲▲

        // 2. 誰もいないロックされたルームを解除
        $stmt_find_locked = $pdo->prepare("SELECT cr.room_id FROM chat_rooms cr WHERE cr.is_locked = 1 AND NOT EXISTS (SELECT 1 FROM online_status os WHERE os.room_id = cr.room_id)");
        $stmt_find_locked->execute();
        $rooms_to_unlock = $stmt_find_locked->fetchAll(PDO::FETCH_COLUMN, 0);
        if (!empty($rooms_to_unlock)) {
            $placeholders = implode(',', array_fill(0, count($rooms_to_unlock), '?'));
            $pdo->prepare("UPDATE chat_rooms SET is_locked = 0 WHERE room_id IN ($placeholders)")->execute($rooms_to_unlock);
        }

        // 3. 有効期限切れのチャットログを削除
        $pdo->prepare("DELETE FROM chat_log3 WHERE expires_at IS NOT NULL AND expires_at < NOW()")->execute();

        $pdo->commit();

        // 4. ルーム一覧のキャッシュをクリア
        clear_room_list_cache();

        return true;
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("run_maintenance() failed: " . $e->getMessage());
        return false;
    }
}

 /**
  * 文字列内のURLを<a>タグに変換する
  * @param string $text 変換対象のテキスト
  * @return string 変換後のHTML
  */
 function make_links_clickable($text)
 {
     // config.phpで機能が無効化されていれば、何もせずに元のテキストを返す
     if (!defined('ENABLE_AUTO_LINK') || ENABLE_AUTO_LINK === false) {
         return h($text);
     }

     // XSS対策として先にエスケープ
     $text = h($text);

     // URLを検出するための正規表現
     $pattern = '/(https?:\/\/[^\s<>"\'()]+)/';
     $replacement = '<a href="$1" target="_blank" rel="noopener noreferrer">$1</a>';

     return preg_replace($pattern, $replacement, $text);
 }
