<?php

// FILE: ajax.php (PRODUCTION READY - FULLY REVISED)
ini_set('display_errors', 0);
ini_set('log_errors', 1);
ini_set('error_log', __DIR__ . '/ajax_error.txt');
error_reporting(E_ALL);
date_default_timezone_set('Asia/Tokyo');

require_once __DIR__ . '/config.php';
session_start(['cookie_samesite' => 'None','cookie_secure' => true,'cookie_httponly' => true]);
require_once __DIR__ . '/db.php';
require_once __DIR__ . '/functions.php';

if (!function_exists('validate_csrf_token_ajax')) {
    function validate_csrf_token_ajax()
    {
        if (session_status() !== PHP_SESSION_ACTIVE) {
            error_log("CSRF validation for Ajax skipped: Session not active.");
            return;
        }
        $post_token = $_POST['csrf_token'] ?? '';
        $session_token = $_SESSION['csrf_token'] ?? null;
        if (empty($post_token) || empty($session_token) || !hash_equals($session_token, $post_token)) {
            unset($_SESSION['csrf_token']);
            generate_csrf_token();
            header('Content-Type: application/json; charset=utf-8', true, 403);
            echo json_encode([
                'status' => 'error',
                'message' => 'ページの有効期限が切れました。ページを更新してもう一度お試しください。',
                'new_csrf_token' => $_SESSION['csrf_token']
            ], JSON_UNESCAPED_UNICODE);
            exit;
        }
    }
}
$action = $_REQUEST['action'] ?? '';
if (!isset($_SESSION['user_id'])) {
    header('Content-Type: application/json; charset=utf-8', true, 401);
    echo json_encode(['status' => 'error', 'message' => 'Not logged in'], JSON_UNESCAPED_UNICODE);
    exit;
}
$pdo = get_pdo();
$user_id = $_SESSION['user_id'];

function get_messages($pdo, $user_id)
{
    session_write_close();
    $room_id = $_GET['room_id'] ?? 0;
    if (empty($room_id)) {
        throw new Exception('Invalid room ID.');
    }
    $last_log_id = $_GET['last_log_id'] ?? 0;
    $stmt_beep = $pdo->prepare("SELECT beep_enabled FROM users3 WHERE user_id = :user_id");
    $stmt_beep->execute([':user_id' => $user_id]);
    $current_user_beep_enabled = (int)$stmt_beep->fetchColumn();
    $online_check_threshold = date('Y-m-d H:i:s', time() - ONLINE_THRESHOLD_SECONDS);
    $sql_msg = "
		SELECT 
			l.log_id, l.user_id, l.message, l.file_name, l.file_path, l.original_file_name, 
			DATE_FORMAT(l.created_at, '%m/%d %H:%i') as created_at, 
			u.user_name, u.icon_image, u.height, u.weight, u.age, 
			ut.usertype, ut.typecolor, 
			l.is_private, l.to_user_id, tu.user_name as to_user_name, l.is_system,
			CASE WHEN os.last_active > :online_threshold THEN 1 ELSE 0 END AS is_online
		FROM chat_log3 l 
		LEFT JOIN users3 u ON l.user_id = u.user_id
		LEFT JOIN online_status os ON l.user_id = os.user_id AND os.room_id = l.room_id
		LEFT JOIN usertype ut ON u.usertype = ut.usertype 
		LEFT JOIN users3 tu ON l.to_user_id = tu.user_id 
		WHERE l.room_id = :room_id AND l.log_id > :last_log_id AND l.deleted = 0 
		  AND (l.is_system = 1 OR l.is_private = 0 OR l.user_id = :user_id OR l.to_user_id = :user_id_too)
		ORDER BY l.log_id ASC
	";
    $stmt_msg = $pdo->prepare($sql_msg);
    $stmt_msg->execute([':online_threshold' => $online_check_threshold,':room_id' => $room_id, ':last_log_id' => $last_log_id, ':user_id' => $user_id, ':user_id_too' => $user_id]);
    $messages = $stmt_msg->fetchAll();
    $last_check_time = date('Y-m-d H:i:s', time() - (CHAT_POLLING_INTERVAL_MS / 1000 + 5));
    $sql_del = "SELECT log_id FROM chat_log3 WHERE room_id = :room_id AND updated_at > :last_check_time AND deleted = 1 AND (is_private = 0 OR user_id = :user_id OR to_user_id = :user_id_too)";
    $stmt_del = $pdo->prepare($sql_del);
    $stmt_del->execute([':room_id' => $room_id, ':last_check_time' => $last_check_time, ':user_id' => $user_id, ':user_id_too' => $user_id]);
    $deleted_ids = $stmt_del->fetchAll(PDO::FETCH_COLUMN, 0);
    $user_list_timestamp = get_user_list_update_timestamp($room_id);
    session_start();
    $csrf_token = $_SESSION['csrf_token'] ?? '';
    session_write_close();
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => 'success', 'beep_enabled' => $current_user_beep_enabled, 'messages' => $messages, 'deleted_ids' => $deleted_ids, 'user_list_timestamp' => $user_list_timestamp, 'csrf_token' => $csrf_token], JSON_UNESCAPED_UNICODE);
}

function get_users_and_room_status($pdo, $user_id)
{
    session_write_close();
    $room_id = $_GET['room_id'] ?? 0;
    if (empty($room_id)) {
        throw new Exception('Invalid room ID.');
    }

    $pdo->prepare("INSERT INTO online_status (user_id, room_id, last_active) VALUES (:uid, :rid, NOW()) ON DUPLICATE KEY UPDATE last_active = NOW()")
      ->execute([':uid' => $user_id, ':rid' => $room_id]);

    $stmt_room_info = $pdo->prepare("SELECT created_by, is_locked, flag_meet, flag_chatplay, flag_onlyreal, flag_friend FROM chat_rooms WHERE room_id = :room_id");
    $stmt_room_info->execute([':room_id' => $room_id]);
    $room_data = $stmt_room_info->fetch();
    if ($room_data === false) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'room_closed' => true], JSON_UNESCAPED_UNICODE);
        exit;
    }
    $creator_id = $room_data['created_by'];

    $stmt_user_status = $pdo->prepare("SELECT deny FROM users3 WHERE user_id = :user_id");
    $stmt_user_status->execute([':user_id' => $user_id]);
    if ($stmt_user_status->fetchColumn() == 1) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'is_denied' => true], JSON_UNESCAPED_UNICODE);
        exit;
    }

    $stmt_users = $pdo->prepare("
		SELECT 
			u.user_id, u.user_name,
			CASE
				WHEN os.last_active >= DATE_SUB(NOW(), INTERVAL :threshold SECOND) THEN 1
				ELSE 0
			END AS is_online,
			os.created_at as entered_at,
			os.last_active as last_active_str,
			os.message_cnt,
			IFNULL(ts.is_typing, 0) as is_typing,
			u.icon_image, u.height, u.weight, u.age, u.areaname, 
			u.tastes, u.wants, u.hopes, u.profile_message, u.is_spectator,
			ut.usertype, ut.typecolor, u.flag_meet, u.flag_chatplay, u.flag_onlyreal, u.flag_friend, u.device_type
		FROM online_status os
		LEFT JOIN users3 u ON os.user_id = u.user_id
		LEFT JOIN usertype ut ON u.usertype = ut.usertype
		LEFT JOIN typing_status ts ON os.user_id = ts.user_id AND os.room_id = ts.room_id
		WHERE os.room_id = :room_id AND (u.deny = 0 OR u.deny IS NULL)
		ORDER BY os.created_at ASC
	");
    $stmt_users->execute([
        ':room_id' => $room_id,
        ':threshold' => ONLINE_THRESHOLD_SECONDS // PHPの定数を直接渡す
    ]);
    $all_users_in_room = $stmt_users->fetchAll(PDO::FETCH_ASSOC);

    $users = [];
    $spectator_count = 0;

    foreach ($all_users_in_room as $user_data) {
        // 自分自身のオンライン状態は常にtrue（オンライン）として上書きする
        if ($user_data['user_id'] == $user_id) {
            $user_data['is_online'] = 1;
        }

        $user_data['is_creator'] = ($user_data['user_id'] == $creator_id);
        $user_data['area_color'] = defined('AREA_NAME_COLOR') ? AREA_NAME_COLOR : 'royalblue';

        if ($user_data['is_spectator'] == 1) {
            if ($user_data['is_online']) {
                $spectator_count++;
            }
            if ($user_data['user_id'] == $user_id) {
                $users[] = $user_data;
            }
        } else {
            $users[] = $user_data;
        }
    }

    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => 'success', 'room_closed' => false, 'is_denied' => false, 'is_locked' => $room_data['is_locked'], 'users' => $users, 'spectator_count' => $spectator_count], JSON_UNESCAPED_UNICODE);
}

function post_message($pdo, $user_id)
{
    validate_csrf_token_ajax();
    $room_id = $_POST['room_id'] ?? 0;
    $message = trim($_POST['message'] ?? '');
    $to_user_id = empty($_POST['to_user_id']) ? null : (int)$_POST['to_user_id'];
    $is_private = $to_user_id ? 1 : 0;
    $pdo->prepare("UPDATE online_status SET message_cnt = message_cnt + 1, last_active = NOW() WHERE user_id = :user_id AND room_id = :room_id")->execute([':user_id' => $user_id, ':room_id' => $room_id]);
    $stmt_user_info = $pdo->prepare("SELECT u.deny, u.log_expire_days, u.user_name, u.icon_image, u.height, u.weight, u.age, ut.usertype, ut.typecolor, u.is_spectator FROM users3 u LEFT JOIN usertype ut ON u.usertype = ut.usertype WHERE u.user_id = :user_id");
    $stmt_user_info->execute([':user_id' => $user_id]);
    $user_info = $stmt_user_info->fetch();
    if (!$user_info || $user_info['deny'] == 1) {
        throw new Exception('このチャットルームは利用できません');
    }
    $to_user_name = null;
    if ($to_user_id) {
        $stmt_to_user = $pdo->prepare("SELECT user_name FROM users3 WHERE user_id = :to_user_id");
        $stmt_to_user->execute([':to_user_id' => $to_user_id]);
        $to_user_name = $stmt_to_user->fetchColumn();
    }
    $file_uploaded = isset($_FILES['file']) && $_FILES['file']['error'] == UPLOAD_ERR_OK;
    if (empty($message) && !$file_uploaded) {
        throw new Exception('メッセージまたはファイルが空です。');
    }
    $file_name = null;
    $file_path = null;
    $original_file_name = null;
    if ($file_uploaded) {
        if ($_FILES['file']['size'] > 2097152) {
            throw new Exception('ファイルサイズが2MBを超えています。');
        }
        $upload_dir = 'uploads/';
        if (!is_dir($upload_dir)) {
            if (!mkdir($upload_dir, 0777, true)) {
                throw new Exception('アップロード用ディレクトリの作成に失敗しました。');
            }
        }
        $original_file_name = basename($_FILES['file']['name']);
        $file_extension = strtolower(pathinfo($original_file_name, PATHINFO_EXTENSION));
        $tmp_name = $_FILES['file']['tmp_name'];
        $allowed_extensions = ['jpg', 'jpeg', 'png', 'gif', 'pdf', 'doc', 'docx', 'xls', 'xlsx', 'txt'];
        if (!in_array($file_extension, $allowed_extensions)) {
            throw new Exception('このファイル形式はアップロードできません。');
        }
        $new_file_name = uniqid('', true) . '.' . $file_extension;
        $destination = $upload_dir . $new_file_name;
        if (!move_uploaded_file($tmp_name, $destination)) {
            throw new Exception('ファイルのアップロード処理中にエラーが発生しました。');
        }
        $file_name = $new_file_name;
        $file_path = $destination;
    }
    $system_message_data = null;
    if ($user_info['is_spectator'] == 1) {
        $pdo->prepare("UPDATE users3 SET is_spectator = 0 WHERE user_id = :user_id")->execute([':user_id' => $user_id]);
        $system_message_text = h($user_info['user_name']) . 'さんの覗き見モードが通常モードに変わりました';
        $stmt = $pdo->prepare("INSERT INTO chat_log3 (room_id, message, is_system) VALUES (:room_id, :message, 1)");
        $stmt->execute([':room_id' => $room_id, ':message' => $system_message_text]);
        $system_log_id = $pdo->lastInsertId();
        $system_message_data = ['log_id' => $system_log_id, 'message' => $system_message_text, 'is_system' => 1, 'created_at' => date('m/d H:i')];
        touch_user_list_update_timestamp($room_id);
        clear_room_list_cache();
    }
    $log_expire_days = (int)$user_info['log_expire_days'];
    $expires_at = null;
    if ($log_expire_days >= 1) {
        $expires_at = date('Y-m-d H:i:s', strtotime("+" . $log_expire_days . " day"));
    } elseif ($log_expire_days === 0) {
        $expires_at = date('Y-m-d H:i:s', time() + (AUTO_LOGOUT_SECONDS * 2));
    }
    $sql_insert = "INSERT INTO chat_log3 (room_id, user_id, message, file_name, file_path, original_file_name, is_private, to_user_id, expires_at) VALUES (:room_id, :user_id, :message, :file_name, :file_path, :original_file_name, :is_private, :to_user_id, :expires_at)";
    $stmt_insert = $pdo->prepare($sql_insert);
    $stmt_insert->execute([ ':room_id' => $room_id, ':user_id' => $user_id, ':message' => $message, ':file_name' => $file_name, ':file_path' => $file_path, ':original_file_name' => $original_file_name, ':is_private' => $is_private, ':to_user_id' => $to_user_id, ':expires_at' => $expires_at ]);
    $new_log_id = $pdo->lastInsertId();
    $message_data = ['log_id' => $new_log_id, 'user_id' => $user_id, 'message' => $message, 'file_name' => $file_name, 'file_path' => $file_path, 'original_file_name' => $original_file_name, 'created_at' => date('m/d H:i'), 'created_timestamp' => time(), 'user_name' => $user_info['user_name'], 'icon_image' => $user_info['icon_image'], 'height' => $user_info['height'], 'weight' => $user_info['weight'], 'age' => $user_info['age'], 'usertype' => $user_info['usertype'], 'typecolor' => $user_info['typecolor'], 'is_private' => $is_private, 'to_user_id' => $to_user_id, 'to_user_name' => $to_user_name, 'is_system' => 0, 'is_online' => 1];
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => 'success', 'message_data' => $message_data, 'system_message_data' => $system_message_data], JSON_UNESCAPED_UNICODE);
    if (function_exists('fastcgi_finish_request')) {
        fastcgi_finish_request();
    }
    try {
        $pdo->prepare("UPDATE chat_rooms SET updated_at = NOW() WHERE room_id = :room_id")->execute([':room_id' => $room_id]);
        $message_to_log = !empty($message) ? $message : "[File: ".$original_file_name."]";
        $pdo->prepare("UPDATE users3 SET last_message = :last_message WHERE user_id = :user_id")->execute([':last_message' => $message_to_log, ':user_id' => $user_id]);
    } catch (Exception $e) {
        error_log("post_message background processing failed: " . $e->getMessage());
    }
}
function update_typing($pdo, $user_id)
{
    validate_csrf_token_ajax();
    if (defined('ENABLE_TYPING_NOTIFICATION') && ENABLE_TYPING_NOTIFICATION === true) {
        $is_typing = (($_POST['status'] ?? 'stopped') === 'typing') ? 1 : 0;
        $room_id = filter_input(INPUT_POST, 'room_id', FILTER_VALIDATE_INT);
        if ($room_id) {
            $sql = "INSERT INTO typing_status (user_id, room_id, is_typing) VALUES (:user_id, :room_id, :is_typing) ON DUPLICATE KEY UPDATE room_id = VALUES(room_id), is_typing = VALUES(is_typing)";
            $stmt = $pdo->prepare($sql);
            $stmt->execute([':user_id' => $user_id, ':room_id' => $room_id, ':is_typing' => $is_typing]);
        }
    }
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => 'success'], JSON_UNESCAPED_UNICODE);
}
function delete_message($pdo, $user_id)
{
    validate_csrf_token_ajax();
    $log_id = $_POST['log_id'] ?? 0;
    if (empty($log_id)) {
        throw new Exception('Invalid log ID.');
    }
    $stmt = $pdo->prepare("SELECT user_id, file_path FROM chat_log3 WHERE log_id = :log_id AND deleted = 0");
    $stmt->execute([':log_id' => $log_id]);
    $log = $stmt->fetch();
    if (!$log) {
        throw new Exception('メッセージが存在しないか、すでに削除されています。');
    }
    if ($log['user_id'] != $user_id) {
        throw new Exception('このメッセージを削除する権限がありません。');
    }
    $php_current_time = date('Y-m-d H:i:s');
    $pdo->prepare("UPDATE chat_log3 SET deleted = 1, message = '（削除されました）', file_path = NULL, file_name = NULL, original_file_name = NULL, updated_at = :now WHERE log_id = :log_id")->execute([':now' => $php_current_time, ':log_id' => $log_id]);
    if (!empty($log['file_path']) && file_exists($log['file_path'])) {
        @unlink($log['file_path']);
    }
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['status' => 'success'], JSON_UNESCAPED_UNICODE);
}
function kick_user($pdo, $requesting_user_id)
{
    validate_csrf_token_ajax();
    $room_id = $_POST['room_id'] ?? 0;
    $target_user_id = $_POST['target_user_id'] ?? 0;
    if (empty($room_id) || empty($target_user_id) || $requesting_user_id == $target_user_id) {
        throw new Exception('無効なリクエストです。');
    }
    $stmt_check = $pdo->prepare("SELECT created_by FROM chat_rooms WHERE room_id = :room_id");
    $stmt_check->execute([':room_id' => $room_id]);
    $creator_id = $stmt_check->fetchColumn();
    if ($creator_id === false || $creator_id != $requesting_user_id) {
        throw new Exception('この操作を行う権限がありません。');
    }
    $php_current_time = date('Y-m-d H:i:s');
    $stmt_update = $pdo->prepare("UPDATE users3 SET deny = 1, last_active_room_id = NULL, denied_at = :now WHERE user_id = :user_id");
    $stmt_update->execute([':now' => $php_current_time, ':user_id' => $target_user_id]);
    if ($stmt_update->rowCount() > 0) {
        clear_room_list_cache();
        touch_user_list_update_timestamp($room_id);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'message' => 'ユーザーを退室させました。'], JSON_UNESCAPED_UNICODE);
    } else {
        throw new Exception('ユーザーの退室処理に失敗しました。');
    }
}
function toggle_room_lock($pdo, $user_id)
{
    validate_csrf_token_ajax();
    $room_id = $_POST['room_id'] ?? 0;
    if (empty($room_id)) {
        throw new Exception('ルームIDが指定されていません。');
    }
    $pdo->beginTransaction();
    try {
        $stmt_check = $pdo->prepare("SELECT created_by, is_locked FROM chat_rooms WHERE room_id = :room_id FOR UPDATE");
        $stmt_check->execute([':room_id' => $room_id]);
        $room = $stmt_check->fetch(PDO::FETCH_ASSOC);
        if (!$room) {
            throw new Exception('ルームが存在しません。');
        }
        if ($room['created_by'] != $user_id) {
            throw new Exception('この操作を行う権限がありません。');
        }
        $new_lock_status = ($room['is_locked'] == 1) ? 0 : 1;
        $stmt_update_room = $pdo->prepare("UPDATE chat_rooms SET is_locked = :is_locked WHERE room_id = :room_id");
        $stmt_update_room->execute([':is_locked' => $new_lock_status, ':room_id' => $room_id]);
        $user_name = $_SESSION['user_name'] ?? '管理者';
        $system_message = ($new_lock_status == 1) ? h($user_name) . 'さんが入室制限を設定しました' : h($user_name) . 'さんが入室制限を解除しました';
        $stmt_system_message = $pdo->prepare("INSERT INTO chat_log3 (room_id, message, is_system) VALUES (:room_id, :msg, 1)");
        $stmt_system_message->execute([':room_id' => $room_id, ':msg' => $system_message]);
        $pdo->commit();
        touch_user_list_update_timestamp($room_id);
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'is_locked' => $new_lock_status], JSON_UNESCAPED_UNICODE);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}
function report_image($pdo, $reporter_user_id)
{
    validate_csrf_token_ajax();
    $log_id = $_POST['log_id'] ?? 0;
    if (empty($log_id)) {
        throw new Exception('Invalid log ID.');
    }
    $pdo->beginTransaction();
    try {
        $stmt_get_log = $pdo->prepare("SELECT room_id, user_id, file_path, file_name FROM chat_log3 WHERE log_id = :log_id AND file_path IS NOT NULL AND deleted = 0");
        $stmt_get_log->execute([':log_id' => $log_id]);
        $log = $stmt_get_log->fetch();
        if (!$log) {
            throw new Exception('通報対象の画像が存在しないか、すでに処理されています。');
        }
        if ($reporter_user_id == $log['user_id']) {
            throw new Exception('自分自身の画像を報告することはできません。');
        }
        $stmt_insert_report = $pdo->prepare("INSERT INTO reports (log_id, room_id, reporter_user_id, reported_user_id, image_file_path) VALUES (:log_id, :room_id, :reporter_user_id, :reported_user_id, :image_file_path)");
        $stmt_insert_report->execute([':log_id' => $log_id, ':room_id' => $log['room_id'], ':reporter_user_id' => $reporter_user_id, ':reported_user_id' => $log['user_id'], ':image_file_path' => $log['file_path']]);
        $report_dir = 'uploads/reports/';
        if (!is_dir($report_dir)) {
            if (!mkdir($report_dir, 0777, true)) {
                throw new Exception('通報用ディレクトリの作成に失敗しました。');
            }
        }
        $new_file_path = $report_dir . $log['file_name'];
        if (file_exists($log['file_path'])) {
            @rename($log['file_path'], $new_file_path);
        }
        $stmt_update_log = $pdo->prepare("UPDATE chat_log3 SET file_path = NULL, message = CONCAT(IFNULL(message, ''), '\n（画像は通報により削除されました）') WHERE log_id = :log_id");
        $stmt_update_log->execute([':log_id' => $log_id]);
        $pdo->commit();
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success'], JSON_UNESCAPED_UNICODE);
    } catch (Exception $e) {
        $pdo->rollBack();
        throw $e;
    }
}
function close_room($pdo, $user_id)
{
    validate_csrf_token_ajax();
    $room_id_to_close = $_POST['room_id'] ?? 0;
    if (empty($room_id_to_close)) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'error', 'message' => '無効なリクエストです。'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    $stmt_check = $pdo->prepare("SELECT created_by, fixed FROM chat_rooms WHERE room_id = :room_id");
    $stmt_check->execute([':room_id' => $room_id_to_close]);
    $room = $stmt_check->fetch();
    if (!$room) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'error', 'message' => '対象のルームが存在しません。'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    if ($room['created_by'] != $user_id) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'error', 'message' => 'このルームを閉鎖する権限がありません。'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    if ($room['fixed'] == 1) {
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'error', 'message' => 'このルームは固定部屋のため、閉鎖できません。'], JSON_UNESCAPED_UNICODE);
        exit;
    }
    $pdo->beginTransaction();
    try {
        $stmt_get_files = $pdo->prepare("SELECT file_path FROM chat_log3 WHERE room_id = :room_id AND file_path IS NOT NULL");
        $stmt_get_files->execute([':room_id' => $room_id_to_close]);
        $files_to_delete = $stmt_get_files->fetchAll(PDO::FETCH_COLUMN, 0);
        $pdo->prepare("DELETE FROM chat_log3 WHERE room_id = :room_id")->execute([':room_id' => $room_id_to_close]);
        $pdo->prepare("DELETE FROM chat_rooms WHERE room_id = :room_id")->execute([':room_id' => $room_id_to_close]);
        $pdo->commit();
        foreach ($files_to_delete as $file) {
            if (!empty($file) && file_exists($file)) {
                @unlink($file);
            }
        }
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'message' => 'ルームを閉鎖しました。'], JSON_UNESCAPED_UNICODE);
    } catch (Exception $e) {
        $pdo->rollBack();
        error_log("Room closing failed: " . $e->getMessage());
        header('Content-Type: application/json; charset=utf-8', true, 500);
        echo json_encode(['status' => 'error', 'message' => 'データベース処理中にエラーが発生しました。'], JSON_UNESCAPED_UNICODE);
    }
}
function update_profile($pdo, $user_id)
{
    validate_csrf_token_ajax();
    try {
        $pdo->beginTransaction();
        $user_name = trim($_POST['user_name'] ?? '');
        if (empty($user_name)) {
            throw new Exception('ユーザー名は必須です。');
        }
        $height = trim($_POST['height'] ?? '');
        $weight = trim($_POST['weight'] ?? '');
        $age = trim($_POST['age'] ?? '');
        if (!empty($height) && !ctype_digit($height)) {
            throw new Exception('身長は半角数字で入力してください。');
        }
        if (!empty($weight) && !ctype_digit($weight)) {
            throw new Exception('体重は半角数字で入力してください。');
        }
        if (!empty($age) && !ctype_digit($age)) {
            throw new Exception('年齢は半角数字で入力してください。');
        }
        $profile_message = trim($_POST['profile_message'] ?? '');
        $update_fields = [
            ':user_name' => $user_name, ':usertype' => trim($_POST['usertype'] ?? ''), ':height' => $height, ':weight' => $weight, ':age' => $age, ':areaname' => trim($_POST['areaname'] ?? ''),
            ':profile_message' => $profile_message, ':tastes' => implode(',', $_POST['tastes'] ?? []), ':wants' => implode(',', $_POST['wants'] ?? []), ':hopes' => implode(',', $_POST['hopes'] ?? []),
            ':beep_enabled' => isset($_POST['beep_enabled']) ? 1 : 0, ':log_expire_days' => (int)($_POST['log_expire_days'] ?? 1),
            ':flag_meet' => isset($_POST['flag_meet']) ? 1 : 0, ':flag_chatplay' => isset($_POST['flag_chatplay']) ? 1 : 0,
            ':flag_onlyreal' => isset($_POST['flag_onlyreal']) ? 1 : 0, ':flag_friend' => isset($_POST['flag_friend']) ? 1 : 0,
            ':user_id' => $user_id
        ];
        $upload_dir = 'uploads/';
        $stmt_old_icon = $pdo->prepare("SELECT icon_image FROM users3 WHERE user_id = :user_id");
        $stmt_old_icon->execute([':user_id' => $user_id]);
        $old_icon_file = $stmt_old_icon->fetchColumn();
        if (isset($_POST['delete_icon']) && $_POST['delete_icon'] == '1') {
            $update_fields[':icon_image'] = DEFAULT_ICON_FILENAME;
            if ($old_icon_file && $old_icon_file !== DEFAULT_ICON_FILENAME && file_exists($upload_dir . $old_icon_file)) {
                @unlink($upload_dir . $old_icon_file);
            }
        } elseif (isset($_FILES['icon_image']) && $_FILES['icon_image']['error'] == UPLOAD_ERR_OK) {
            if ($_FILES['icon_image']['size'] > 2097152) {
                throw new Exception('ファイルサイズが2MBを超えています。');
            }
            $allowed_types = ['image/jpeg', 'image/png', 'image/gif'];
            if (!in_array($_FILES['icon_image']['type'], $allowed_types)) {
                throw new Exception('許可されていないファイル形式です。');
            }
            $file_extension = strtolower(pathinfo($_FILES['icon_image']['name'], PATHINFO_EXTENSION));
            $new_file_name = $user_id . '_' . uniqid() . '.' . $file_extension;
            $destination = $upload_dir . $new_file_name;
            if (resize_image($_FILES['icon_image']['tmp_name'], $destination, ['max_width' => MAX_PROFILE_IMAGE_WIDTH])) {
                $update_fields[':icon_image'] = $new_file_name;
                if ($old_icon_file && $old_icon_file !== DEFAULT_ICON_FILENAME && file_exists($upload_dir . $old_icon_file)) {
                    @unlink($upload_dir . $old_icon_file);
                }
            } else {
                throw new Exception('アイコンのアップロード処理に失敗しました。');
            }
        }
        $set_clause_parts = [];
        foreach ($update_fields as $key => $value) {
            if ($key !== ':user_id') {
                $set_clause_parts[] = ltrim($key, ':') . " = " . $key;
            }
        }
        $sql = "UPDATE users3 SET " . implode(', ', $set_clause_parts) . " WHERE user_id = :user_id";
        $pdo->prepare($sql)->execute($update_fields);
        $target_room_id = $_SESSION['room_id'] ?? null;
        if ($target_room_id) {
            $pdo->prepare("UPDATE chat_rooms SET description = :desc WHERE room_id = :room_id AND created_by = :user_id")->execute([':desc' => $profile_message, ':room_id' => $target_room_id, ':user_id' => $user_id]);
            $system_message = h($user_name) . 'さんのプロフィールが更新されました';
            $pdo->prepare("INSERT INTO chat_log3 (room_id, message, is_system) VALUES (:room_id, :msg, 1)")->execute([':room_id' => $target_room_id, ':msg' => $system_message]);
            touch_user_list_update_timestamp($target_room_id);
            clear_room_list_cache();
        }
        $pdo->commit();
        $_SESSION['user_name'] = $user_name;
        header('Content-Type: application/json; charset=utf-8');
        echo json_encode(['status' => 'success', 'message' => 'プロフィールを更新しました。', 'new_user_name' => $user_name], JSON_UNESCAPED_UNICODE);
    } catch (Exception $e) {
        if ($pdo->inTransaction()) {
            $pdo->rollBack();
        }
        error_log("Profile update failed in ajax: " . $e->getMessage());
        header('Content-Type: application/json; charset=utf-8', true, 400);
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()], JSON_UNESCAPED_UNICODE);
    }
}

try {
    switch ($action) {
        case 'get_messages': get_messages($pdo, $user_id); break;
        case 'get_users_and_room_status': get_users_and_room_status($pdo, $user_id); break;
        case 'post_message': post_message($pdo, $user_id); break;
        case 'update_typing': update_typing($pdo, $user_id); break;
        case 'delete_message': delete_message($pdo, $user_id); break;
        case 'kick_user': kick_user($pdo, $user_id); break;
        case 'toggle_room_lock': toggle_room_lock($pdo, $user_id); break;
        case 'report_image': report_image($pdo, $user_id); break;
        case 'close_room': close_room($pdo, $user_id); break;
        case 'update_profile': update_profile($pdo, $user_id); break;
        default:
            header('Content-Type: application/json; charset=utf-8');
            echo json_encode(['status' => 'error', 'message' => 'Invalid action specified.'], JSON_UNESCAPED_UNICODE);
            break;
    }
} catch (Throwable $e) {
    header('Content-Type: application/json; charset=utf-8', true, 500);
    echo json_encode(['status' => 'error', 'message' => 'An internal server error occurred.', 'error_details' => ['message' => $e->getMessage(),'file' => $e->getFile(),'line' => $e->getLine()]], JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
}
